<?php
defined( 'ABSPATH' ) || exit;

class WC_Gateway_ePay extends WC_Payment_Gateway {

    public function __construct() {
        $this->id                 = 'epay';
        // Load logo from main application
        $this->payment_url        = $this->get_option( 'payment_url', 'https://secure.epay.com.bd/' );
        $base_url                 = str_replace( 'secure.', '', $this->payment_url ); // Get main domain
        $this->icon               = apply_filters( 'woocommerce_epay_icon', $base_url . 'public/assets/images/logo.png' );
        
        $this->has_fields         = false;
        $this->method_title       = __( 'ePay', 'epay-payment-gateway' );
        $this->method_description = __( 'Accept payments securely via ePay Payment Gateway.', 'epay-payment-gateway' );

        // Load the settings
        $this->init_form_fields();
        $this->init_settings();

        // Define user set variables
        $this->title        = $this->get_option( 'title' );
        $this->description  = $this->get_option( 'description' );
        $this->enabled      = $this->get_option( 'enabled' );
        $this->payment_url  = $this->get_option( 'payment_url', 'https://secure.epay.com.bd/' );
        $this->brand_key    = $this->get_option( 'brand_key' );
        $this->api_key      = $this->get_option( 'api_key' );
        $this->api_secret   = $this->get_option( 'api_secret' );

        // Actions
        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
        add_action( 'woocommerce_api_wc_gateway_epay', array( $this, 'handle_callback' ) );
        
        // Admin Scripts for Connect Logic
        add_action( 'admin_footer', array( $this, 'admin_scripts' ) );
    }

    /**
     * Initialize Gateway Settings Form Fields
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title'   => __( 'Enable/Disable', 'epay-payment-gateway' ),
                'type'    => 'checkbox',
                'label'   => __( 'Enable ePay Payment', 'epay-payment-gateway' ),
                'default' => 'yes'
            ),
            'title' => array(
                'title'       => __( 'Title', 'epay-payment-gateway' ),
                'type'        => 'text',
                'description' => __( 'This controls the title which the user sees during checkout.', 'epay-payment-gateway' ),
                'default'     => __( 'Pay with ePay', 'epay-payment-gateway' ),
                'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __( 'Description', 'epay-payment-gateway' ),
                'type'        => 'textarea',
                'description' => __( 'Payment method description that the customer will see on your checkout.', 'epay-payment-gateway' ),
                'default'     => __( 'Pay securely using your mobile wallet or card via ePay.', 'epay-payment-gateway' ),
            ),
            'payment_url' => array(
                'title'       => __( 'Payment URL', 'epay-payment-gateway' ),
                'type'        => 'text',
                'description' => __( 'Base URL of the payment gateway API.', 'epay-payment-gateway' ),
                'default'     => 'https://secure.epay.com.bd/',
                'desc_tip'    => true,
            ),
            'brand_key' => array(
                'title'       => __( 'Brand Key', 'epay-payment-gateway' ),
                'type'        => 'text',
                'description' => __( 'Enter your Brand Key to connect.', 'epay-payment-gateway' ),
                'default'     => '',
                'desc_tip'    => true,
            ),
            'api_key' => array(
                'title'       => __( 'API Key', 'epay-payment-gateway' ),
                'type'        => 'password', // Hidden for security
                'description' => __( 'Auto-filled after connection.', 'epay-payment-gateway' ),
                'default'     => '',
            ),
            'api_secret' => array(
                'title'       => __( 'API Secret', 'epay-payment-gateway' ),
                'type'        => 'password', // Hidden for security
                'description' => __( 'Auto-filled after connection.', 'epay-payment-gateway' ),
                'default'     => '',
            ),
        );
    }

    /**
     * Custom Admin Options Output to handle "Connect" button
     */
    public function admin_options() {
        ?>
        <h2><?php _e( 'ePay Payment Gateway', 'epay-payment-gateway' ); ?></h2>
        <table class="form-table">
            <?php $this->generate_settings_html(); ?>
            
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label><?php _e( 'Connection Status', 'epay-payment-gateway' ); ?></label>
                </th>
                <td class="forminp">
                    <div id="epay-connection-status">
                        <?php if ( ! empty( $this->api_key ) ) : ?>
                            <span style="color: green; font-weight: bold;">
                                <span class="dashicons dashicons-yes"></span> <?php _e( 'Connected', 'epay-payment-gateway' ); ?>
                            </span>
                            <button type="button" class="button" id="epay-disconnect-btn"><?php _e( 'Disconnect / Reset', 'epay-payment-gateway' ); ?></button>
                        <?php else : ?>
                            <span style="color: red;">
                                <span class="dashicons dashicons-no"></span> <?php _e( 'Not Connected', 'epay-payment-gateway' ); ?>
                            </span>
                            <br><br>
                            <button type="button" class="button button-primary" id="epay-connect-btn"><?php _e( 'Connect with Brand Key', 'epay-payment-gateway' ); ?></button>
                            <span class="spinner" style="float: none; margin-top: 0;"></span>
                            <p class="description"><?php _e( 'Enter your Brand Key above and click Connect to auto-fetch credentials.', 'epay-payment-gateway' ); ?></p>
                        <?php endif; ?>
                    </div>
                    <div id="epay-message" style="margin-top: 10px;"></div>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Admin Scripts for AJAX Connection
     */
    public function admin_scripts() {
        if ( 'woocommerce_page_wc-settings' !== get_current_screen()->id ) {
            return;
        }
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Connect Button
            $('#epay-connect-btn').on('click', function() {
                var brandKey = $('#woocommerce_epay_brand_key').val();
                var paymentUrl = $('#woocommerce_epay_payment_url').val();
                
                if (!brandKey) {
                    alert('Please enter your Brand Key first.');
                    return;
                }

                var $btn = $(this);
                var $spinner = $btn.next('.spinner');
                var $msg = $('#epay-message');

                $btn.prop('disabled', true);
                $spinner.addClass('is-active');
                $msg.html('');

                // API Call to Verify Brand
                $.ajax({
                    url: paymentUrl.replace(/\/$/, "") + '/api/verify-brand',
                    type: 'POST',
                    dataType: 'json',
                    contentType: 'application/json',
                    data: JSON.stringify({ brand_key: brandKey }),
                    success: function(response) {
                        $btn.prop('disabled', false);
                        $spinner.removeClass('is-active');

                        if (response.success) {
                            // Auto-fill fields
                            $('#woocommerce_epay_api_key').val(response.data.api_key);
                            $('#woocommerce_epay_api_secret').val(response.data.api_secret);
                            
                            $msg.html('<span style="color: green;">Connection Successful! Please save changes.</span>');
                            
                            // Trigger save
                            $('button[name="save"]').click();
                        } else {
                            $msg.html('<span style="color: red;">Error: ' + (response.message || 'Invalid Brand Key') + '</span>');
                        }
                    },
                    error: function(xhr, status, error) {
                        $btn.prop('disabled', false);
                        $spinner.removeClass('is-active');
                        $msg.html('<span style="color: red;">Connection Failed. Check Payment URL.</span>');
                        console.error(error);
                    }
                });
            });

            // Disconnect Button
            $('#epay-disconnect-btn').on('click', function() {
                if (confirm('Are you sure you want to disconnect? This will clear your API credentials.')) {
                    $('#woocommerce_epay_api_key').val('');
                    $('#woocommerce_epay_api_secret').val('');
                    $('button[name="save"]').click();
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Process the payment and return the result
     */
    public function process_payment( $order_id ) {
        $order = wc_get_order( $order_id );

        $payload = array(
            'amount'         => $order->get_total(),
            'currency'       => $order->get_currency(),
            'order_id'       => $order_id, // Send simple order ID
            'cus_name'       => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
            'cus_email'      => $order->get_billing_email(),
            'cus_phone'      => $order->get_billing_phone(),
            'success_url'    => WC()->api_request_url( 'WC_Gateway_ePay' ) . '?action=success&oid=' . $order_id,
            'cancel_url'     => $order->get_cancel_order_url(),
            'metadata'       => json_encode(array(
                'wc_order_id' => $order_id
            ))
        );

        $api_url = rtrim( $this->payment_url, '/' ) . '/api/payment/create';

        $response = wp_remote_post( $api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key'    => $this->api_key,
                'X-API-Secret' => $this->api_secret
            ),
            'body'    => json_encode( $payload ),
            'timeout' => 45
        ) );

        if ( is_wp_error( $response ) ) {
            wc_add_notice( __( 'Payment error:', 'epay-payment-gateway' ) . ' ' . $response->get_error_message(), 'error' );
            return;
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['status'] ) && $body['status'] == 'success' && ! empty( $body['payment_url'] ) ) {
            return array(
                'result'   => 'success',
                'redirect' => $body['payment_url']
            );
        } else {
            $msg = isset( $body['message'] ) ? $body['message'] : __( 'Unknown error occurred.', 'epay-payment-gateway' );
            wc_add_notice( __( 'Payment failed:', 'epay-payment-gateway' ) . ' ' . $msg, 'error' );
            return;
        }
    }

    /**
     * Handle Return Callback
     */
    public function handle_callback() {
        if ( isset( $_GET['action'] ) && $_GET['action'] == 'success' && isset( $_GET['payment_id'] ) && isset( $_GET['oid'] ) ) {
            $order_id   = intval( $_GET['oid'] );
            $payment_id = sanitize_text_field( $_GET['payment_id'] );
            $order      = wc_get_order( $order_id );

            if ( ! $order ) {
                wp_die( 'Order not found' );
            }

            // Verify Payment
            $api_url = rtrim( $this->payment_url, '/' ) . '/api/payment/verify/' . $payment_id;
            
            $response = wp_remote_get( $api_url, array(
                'headers' => array(
                    'X-API-Key'    => $this->api_key,
                    'X-API-Secret' => $this->api_secret
                ),
                'timeout' => 45
            ) );

            $body = json_decode( wp_remote_retrieve_body( $response ), true );

            if ( isset( $body['status'] ) && $body['status'] == 'success' && $body['payment_status'] == 'completed' ) {
                $order->payment_complete( $payment_id );
                $order->add_order_note( __( 'Payment completed via ePay. Transaction ID: ' . $payment_id, 'epay-payment-gateway' ) );
                wp_redirect( $this->get_return_url( $order ) );
                exit;
            } else {
                wc_add_notice( __( 'Payment verification failed.', 'epay-payment-gateway' ), 'error' );
                wp_redirect( wc_get_checkout_url() );
                exit;
            }
        }
    }
}
